package de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.impl;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.IAlignedSequence;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.IMatch;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.IPosition;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;

public class AlignedSequence implements IAlignedSequence, Externalizable {
	
	private static final long serialVersionUID = -2757493163971413374L;

	private ContextPositionList[] otherStarts;
	
	private ContextPositionList[] otherEnds;

	private int sequenceId;

	private short length;
	
	private List<Annotation<IAnchoredElement>> sequence;

	private long id;
	
	public AlignedSequence() {
		
	}
	
	
	
	@Override
	public void writeExternal(ObjectOutput out) throws IOException {
		out.writeInt(sequenceId);
		out.writeShort(length);
		out.writeLong(id);
		out.writeInt(otherStarts.length);
		for(int i = 0; i < otherStarts.length; i++) {
			out.writeBoolean(otherStarts[i] != null);
			if(otherStarts[i] != null) {
				otherStarts[i].writeExternal(out);
			}
		}
		out.writeInt(otherEnds.length);
		for(int i = 0; i < otherEnds.length; i++) {
			out.writeBoolean(otherEnds[i] != null);
			if(otherEnds[i] != null) {
				otherEnds[i].writeExternal(out);
			}
		}
		out.writeInt(sequence.size());
		for (Annotation<IAnchoredElement> anno : sequence) {
			anno.writeExternal(out);
		}
	}



	@Override
	public void readExternal(ObjectInput in) throws IOException,
			ClassNotFoundException {
		sequenceId = in.readInt();
		length = in.readShort();
		id = in.readLong();
		int length = in.readInt();
		otherStarts = new ContextPositionList[length];
		for(int i = 0; i < length; i++) {
			if(in.readBoolean()) {
				otherStarts[i] = new ContextPositionList();
				otherStarts[i].readExternal(in);
			}
		}
		length = in.readInt();
		otherEnds = new ContextPositionList[length];
		for(int i = 0; i < length; i++) {
			if(in.readBoolean()) {
				otherEnds[i] = new ContextPositionList();
				otherEnds[i].readExternal(in);
			}
		}
		length = in.readInt();
		sequence = new ArrayList<Annotation<IAnchoredElement>>(length);
		for(int i = 0; i < length; i++) {
			Annotation<IAnchoredElement> anno = new Annotation<IAnchoredElement>();
			anno.readExternal(in);
			sequence.add(anno);
		}
		
	}



	public AlignedSequence(int id, int length) {
		this.sequenceId = id;
		this.length = (short) length;
		otherStarts = new ContextPositionList[length];
		otherEnds = new ContextPositionList[length];
	}
	
	public void trimStartNodes() {
		for (ContextPositionList list : otherStarts) {
			if(list == null) continue;
			list.trim();
		}
	}

	public void trimEndNodes() {
		for (ContextPositionList list : otherEnds) {
			if(list == null) continue;
			list.trim();
		}
	}

	public ContextPositionList[] getStartPositions() {
		return otherStarts;
	}
	
	public ContextPositionList[] getEndPositions() {
		return otherEnds;
	}
	
	public void release() {
		if(endsCache != null) {
			endsCache.clear();
			endsCache = null;
		}
	}
	
	
	private Map<Integer, TreeMap<Short, ContextPosition>> getStartsFor(int start) {
		// Key: Id der ANDEREN Startsequence
		// Value: Map mit Position im Satz der anderen Startsequence und
		// außerdem noch der ContextPosition, um die es grade geht. Viel zu kompliziert!
		ContextPositionList starts = otherStarts[start];
		Iterator<ContextPosition> startIterator = starts.iterator();
		Map<Integer, TreeMap<Short, ContextPosition>> startHypotheses = new HashMap<Integer, TreeMap<Short,ContextPosition>>();
		while(startIterator.hasNext()) {
			ContextPosition startHypothesis = startIterator.next();
			// TODO: Currently ignoring multiple starts in the same sentence!
			TreeMap<Short, ContextPosition> map = startHypotheses.get(startHypothesis.sequence());
			if(map == null) {
				map = new TreeMap<Short, ContextPosition>();
				startHypotheses.put(startHypothesis.sequence(), map);
			}
			map.put(startHypothesis.wordPos(), startHypothesis);
		}
		return startHypotheses;
	}
	
	private transient Map<Integer, Map<Integer, TreeMap<Short, ContextPosition>>> endsCache;
	
	
	private Map<Integer, TreeMap<Short, ContextPosition>> getEndsFor(int end) {
		if(endsCache == null) {
			 endsCache = new HashMap<Integer, Map<Integer, TreeMap<Short,ContextPosition>>>();
		} else {
			Map<Integer, TreeMap<Short, ContextPosition>> cached = endsCache.get(end);
			if(cached != null) {
				return cached;
			}
		}
		ContextPositionList ends = otherEnds[end];
		Map<Integer, TreeMap<Short, ContextPosition>> endHypotheses = new HashMap<Integer, TreeMap<Short,ContextPosition>>();
		Iterator<ContextPosition> endIterator = ends.iterator();
		while(endIterator.hasNext()) {
			ContextPosition endHypothesis = endIterator.next();
			TreeMap<Short, ContextPosition> map = endHypotheses.get(endHypothesis.sequence());
			if(map == null) {
				map = new TreeMap<Short, ContextPosition>();
				endHypotheses.put(endHypothesis.sequence(), map);
			}
			map.put(endHypothesis.wordPos(), endHypothesis);
		}
		endsCache.put(end, endHypotheses);
		return endHypotheses;
	}
	


	
	
	@Override
	public List<IMatch> getMatches(int start, int end) {
		ContextPositionList startPosition = otherStarts[start];
		// 1. Key: Satz-ID
		// 2. Key: Position
		// 3. Key: Kontext-Info
		Map<Integer, TreeMap<Short, ContextPosition>> ends = getEndsFor(end);
		Iterator<ContextPosition> starts = startPosition.iterator();
		List<IMatch> candidates = new ArrayList<IMatch>(256);
		while(starts.hasNext()) {
			ContextPosition sp = starts.next();
			TreeMap<Short, ContextPosition> sequenceCandidates = ends.get(sp.sequence());
			if(sequenceCandidates == null) continue;
			NavigableMap<Short, ContextPosition> wordCandidates = sequenceCandidates.tailMap(sp.wordPos(), false);
			if(wordCandidates.isEmpty()) continue;
			for (ContextPosition ep : wordCandidates.values()) {
				Match ms = new Match(sp, ep);
				candidates.add(ms);
			}
		}
		return candidates;
		
	}
	
	public List<IMatch> getSingleEndMatches(int end, List<int[]> sequenceMatrix) {
		return acceptFromSequenceStartToIndex(getEndsFor(end));
	}
	
	public List<IMatch> getSingleStartMatches(int start, List<int[]> sequenceMatrix) {
		return acceptFromIndexToSequenceEnd(getStartsFor(start), sequenceMatrix);
	}
	
	/**
	 * Returns all matches which start at the given positions and
	 * end at the end of the sequence.
	 * @param starts
	 * @return
	 */
	private List<IMatch> acceptFromIndexToSequenceEnd(
			Map<Integer, TreeMap<Short, ContextPosition>> starts, List<int[]> sequenceMatrix) {
		Iterator<Entry<Integer, TreeMap<Short, ContextPosition>>> entries = starts.entrySet().iterator();
		List<IMatch> matches = new ArrayList<IMatch>();
		while(entries.hasNext()) {
			Entry<Integer, TreeMap<Short, ContextPosition>> entry = entries.next();
			TreeMap<Short, ContextPosition> positions = entry.getValue();
			Set<Entry<Short, ContextPosition>> pSet = positions.entrySet();
			int sequenceId = entry.getKey();
			for (Entry<Short, ContextPosition> hypo : pSet) {
				int wordPos = sequenceMatrix.get(sequenceId).length;
				if(hypo.getValue().wordPos() == wordPos) continue;
				ContextPosition e = new ContextPosition(sequenceId, wordPos, 0);
				matches.add(new Match(hypo.getValue(), e));
			}
		}
		return matches;
	}

	/**
	 * Returns all matches which start at the beginning of the sequence
	 * and end at the given positions.
	 * @param ends
	 * @return
	 */
	private List<IMatch> acceptFromSequenceStartToIndex(
			Map<Integer, TreeMap<Short, ContextPosition>> ends) {
		Iterator<Entry<Integer, TreeMap<Short, ContextPosition>>> entries = ends.entrySet().iterator();
		List<IMatch> matches = new ArrayList<IMatch>();
		while(entries.hasNext()) {
			Entry<Integer, TreeMap<Short, ContextPosition>> entry = entries.next();
			TreeMap<Short, ContextPosition> positions = entry.getValue();
			Set<Entry<Short, ContextPosition>> pSet = positions.entrySet();
			int sequenceId = entry.getKey();
			for (Entry<Short, ContextPosition> hypo : pSet) {
				if(hypo.getValue().wordPos() == -1) continue;
				ContextPosition s = new ContextPosition(sequenceId, -1, 0);
				matches.add(new Match(s, hypo.getValue()));
			}
		}
		return matches;
	}

	public void registerStartHypothesis(IPosition myPosition, Collection<IPosition> sequences, int depth, ArrayList<int[]> sequenceMatrix) {
		register(myPosition, sequences, depth, otherStarts, sequenceMatrix, true);
	}
	
	public void registerEndHypothesis(IPosition myPosition, Collection<IPosition> sequences, int depth, ArrayList<int[]> sequenceMatrix) {
		register(myPosition, sequences, depth, otherEnds, sequenceMatrix, false);
	}

	private void register(final IPosition myPosition, final Collection<IPosition> sequences,
			final int depth, 
			final ContextPositionList[] others, ArrayList<int[]> sequenceMatrix, final boolean isSuffix) {
		final int mySequence = myPosition.getSequenceIndex();
		ContextPositionList other = others[myPosition.getElementIndex()];
		if(other == null) {
			other = new ContextPositionList();
			others[myPosition.getElementIndex()] = other;
		}
		int myType = -1;
		int[] currentSequence = sequenceMatrix.get(mySequence);
		if(isSuffix && currentSequence.length > myPosition.getElementIndex()+1) {
			myType = currentSequence[myPosition.getElementIndex()+1];
		}
		if(!isSuffix && myPosition.getElementIndex() > 0) {
			myType = currentSequence[myPosition.getElementIndex()-1];
		}
		for (IPosition pp : sequences) {
			//Selbst-Alignment ausschließen:
			final int seqId = pp.getSequenceIndex();
			final int wp = pp.getElementIndex();
			if(mySequence == seqId) {
				continue;
			} else {
				if(-1 != myType) {
					if(isSuffix) {
						int[] otherSequence = sequenceMatrix.get(seqId);
						if(otherSequence.length > wp+1) {
							if(myType == otherSequence[wp+1]) {
								continue;
							}
						}
					} else {
						if(wp > 0) {
							int[] otherSequence = sequenceMatrix.get(seqId);
							if(myType == otherSequence[wp-1]) {
								continue;
							}
						}
					}
				}
				other.insertOrUpdate(seqId, wp, depth);
			}
		}
	}

	public int getSequenceId() {
		return sequenceId;
	}

	public TreeSet<Short> getStartIndices() {
		TreeSet<Short> set = new TreeSet<Short>();
		for(int i = 0; i < otherStarts.length; i++) {
			if(otherStarts[i] != null) set.add((short) i); 
		}
		return set;
	}

	public TreeSet<Short> getEndIndices() {
		TreeSet<Short> set = new TreeSet<Short>();
		for(int i = 0; i < otherEnds.length; i++) {
			if(otherEnds[i] != null) set.add((short) i); 
		}
		return set;
	}

	public int getLength() {
		return length;
	}

	public ContextPositionList getStartsAt(int pos) {
		return otherStarts[pos];
	}

	public ContextPositionList getEndsAt(int pos) {
		return otherEnds[pos];
	}

	@Override
	public long getId() {
		return id;
	}

	@Override
	public void setId(long id) {
		this.id = id;
	}

	public void setSequence(List<Annotation<IAnchoredElement>> sequence) {
		this.sequence = sequence;
	}

	@Override
	public List<Annotation<IAnchoredElement>> getWords() {
		return sequence;
	}
	
	

	
}