package de.uni_koeln.spinfo.tesla.component.misc;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

import de.uni_koeln.spinfo.tesla.annotation.adapter.IOutputAdapter;
import de.uni_koeln.spinfo.tesla.annotation.adapter.InputIterator;
import de.uni_koeln.spinfo.tesla.annotation.adapter.TypeMapping;
import de.uni_koeln.spinfo.tesla.annotation.adapter.queryconstraints.Order;
import de.uni_koeln.spinfo.tesla.annotation.adapter.queryconstraints.Range;
import de.uni_koeln.spinfo.tesla.annotation.adapter.tunguska.DefaultTunguskaOutputAdapter;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.IGeneralizedHeuristicConstituent;
import de.uni_koeln.spinfo.tesla.roles.core.access.IAnchoredElementAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.roles.filter.access.IFilterAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.filter.data.IFilter;
import de.uni_koeln.spinfo.tesla.roles.filter.impl.tunguska.access.RangeFilterAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.filter.impl.tunguska.data.AnnotationRangeFilter;
import de.uni_koeln.spinfo.tesla.roles.filter.impl.tunguska.data.Filter;
import de.uni_koeln.spinfo.tesla.roles.parser.access.IConstituentAccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.Result;
import de.uni_koeln.spinfo.tesla.runtime.TeslaComponent;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.AccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Author;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Component;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Configuration;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Description;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.OutputAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.RoleDescription;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Run;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.ThreadMode;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;
import de.uni_koeln.spinfo.tesla.runtime.persistence.DataObject;

@Component(threadMode=ThreadMode.CUSTOM, 
		author=@Author(	author="Stephan Schwiebert", 
						email="sschwieb@spinfo.uni-koeln.de", 
						web="http://www.spinfo.uni-koeln.de/space/sschwieb", 
						organization="Sprachliche Informationsverarbeitung"),
		description=@Description(name="SOG (Analyze)",
								 summary="Generates hypotheses about paradigmatic structures in sequences, " +
								 		"based on the length of the shared context of two phrases." +
								 		" A hypothesis is accepted if at least one of the found contexts matches the configured restrictions.",
								 bigO="not so bad, in general",
								 version="1.0",
								 reusableResults=true))
public class FindProperties extends TeslaComponent {
	
	
    @AccessAdapter(role="de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", name="Sequences", 
    		description="The sequences to align (for instance, sentences)")
    private IAnchoredElementAccessAdapter<IAnchoredElement> sequences;
    
    @AccessAdapter(role="de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", name="Sequence Items",
    		description="The items to align (for instance, words)")
	protected IAnchoredElementAccessAdapter<IAnchoredElement> words;
    
    @AccessAdapter(role="de.uni_koeln.spinfo.tesla.roles.structure.bootstrapping.GeneralizedHeuristicConstituentDetector", name="Constituents", 
    		description="The sequences to align (for instance, sentences)")
    private IConstituentAccessAdapter<IGeneralizedHeuristicConstituent> constituentIn;
    

	@AccessAdapter(role="de.uni_koeln.spinfo.tesla.roles.filter.Filter", name="Filters", min=0, max=-1)
	private List<IFilterAccessAdapter<IAnchoredElement, IFilter<IAnchoredElement>>> filters;
	    
	 @OutputAdapter(dataObject=Filter.class, name="Filter", type=DefaultTunguskaOutputAdapter.class, 
			 accessAdapterImpl=RangeFilterAccessAdapter.class, description="The generated filter, which can be applied on all annotations")
	 @RoleDescription(value="de.uni_koeln.spinfo.tesla.roles.filter.Filter")
	 private IOutputAdapter<Filter<DataObject>> outputAdapter;
	 
	 @Configuration(defaultValue="false", name="Invert matching", min=1, max=1, description="If enabled, an annotation is not accepted if" +
		 		" it matches an annotation from 'Filter'. If disabled, an annotation will be accepted if it " +
		 		" matches an annotation from 'Filter'")
		 private boolean invertMatching = false;
	 
	  private AggregatedResults leftContext = new AggregatedResults(), rightContext= new AggregatedResults(), sentenceLength= new AggregatedResults(), startPosition= new AggregatedResults(), endPosition= new AggregatedResults(), constituentLength= new AggregatedResults(), relativeConstituentsLengths = new AggregatedResults();

	  private int overallConstituents = 0;
	     
    @Run
    public Result run() {
    	InputIterator<String> signals = sequences.getAllSignalIds();
    	signals.registerAsProgressBar(this);
    	while(signals.hasNext()) {
    		String signalId = signals.next();
    		InputIterator<Annotation<IAnchoredElement>> sentences = sequences.getAnchoredElements(Range.forSignal(signalId), Order.ORDER_LEFTANCHOR_ASC, IAnchoredElement.class);
    		AnnotationRangeFilter filter = new AnnotationRangeFilter(invertMatching);
    		while(sentences.hasNext() && !cancelled()) {
    			Annotation<IAnchoredElement> sentence = sentences.next();
    			InputIterator<Annotation<IAnchoredElement>> elements = words.getAnchoredElements(sentence.getRange(false), Order.ORDER_LEFTANCHOR_ASC, IAnchoredElement.class);
    			List<Annotation<IAnchoredElement>> wordList = asList(elements);
    			InputIterator<Annotation<IGeneralizedHeuristicConstituent>> constituents = constituentIn.getAnchoredElements(sentence.getRange(false), Order.ORDER_LEFTANCHOR_ASC, IGeneralizedHeuristicConstituent.class);
    			while(constituents.hasNext() && !cancelled()) {
    				Annotation<IGeneralizedHeuristicConstituent> constituent = constituents.next();
    				boolean rewrite = true;
    				for (IFilterAccessAdapter<IAnchoredElement, IFilter<IAnchoredElement>> f : filters) {
    					Annotation a = constituent;
    					if(!f.matches(a)) {
    						rewrite = false;
    						break;
    					}
    				}
    				if(rewrite) {
    					if(accept(constituent, sentence, wordList)) {
        					filter.add(constituent);
        				}
    				}
    			}
    		}
    		Annotation<AnnotationRangeFilter> anno = outputAdapter.getAnnotationFactory().newSignalAnnotation(signalId, filter, TypeMapping.NONE);
			outputAdapter.store(anno);
    	}
    	try {
    		leftContext.normalizeValues();
			leftContext.toFile("Left Context.txt");
			rightContext.normalizeValues();
			rightContext.toFile("Right Context.txt");
			sentenceLength.normalizeValues();
			sentenceLength.toFile("Sentence Length.txt");
			startPosition.normalizeValues();
			startPosition.toFile("Start Position.txt");
			endPosition.normalizeValues();
			endPosition.toFile("End Position.txt");
			constituentLength.normalizeValues();
			constituentLength.toFile("Constituent Length.txt");
			relativeConstituentsLengths.normalizeValues();
			relativeConstituentsLengths.toFile("Relative Constituent Length.txt");
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
    	return cancelled() ? Result.CANCELLED : Result.OK;
    }
    
    class AggregatedResults {

    	private Map<Integer, Double> values = new TreeMap<Integer, Double>();
    	
		public void increment(int value) {
			Double integer = values.get(value);
			if(integer == null) {
				values.put(value, 1D);
			} else {
				values.put(value, integer+1);
			}
		}

		public void normalizeValues() {
			double count = 0;
			Iterator<Entry<Integer, Double>> iterator = values.entrySet().iterator();
			while(iterator.hasNext()) {
				count += iterator.next().getValue();
			}
			iterator = values.entrySet().iterator();
			while(iterator.hasNext()) {
				Entry<Integer, Double> entry = iterator.next();
				entry.setValue(entry.getValue()/count * 100D);
			}
			
		}

		public void toFile(String string) throws IOException {
			File dir = new File(getComponentName());
			dir.mkdirs();
			File file = new File(dir, string);
			BufferedWriter bw = new BufferedWriter(new FileWriter(file));
			Iterator<Entry<Integer, Double>> entries = values.entrySet().iterator();
			while(entries.hasNext()) {
				Entry<Integer, Double> entry = entries.next();
				bw.write(entry.getKey()+":"+entry.getValue()+"\n");
			}
			bw.close();
		}
    	
    }
      
	private boolean accept(
			Annotation<IGeneralizedHeuristicConstituent> constituent,
			Annotation<IAnchoredElement> sentence,
			List<Annotation<IAnchoredElement>> wordList) {
		overallConstituents++;
		IGeneralizedHeuristicConstituent data = constituent.getDataObject();
		double maxLeftContextLength = data.getAverageLeftContextLength();
		leftContext.increment((int) (maxLeftContextLength*100));
		double maxRightContextLength = data.getAverageRightContextLength();
		rightContext.increment((int) (maxRightContextLength*100));
		int elements = wordList.size();
		sentenceLength.increment(elements);
		int constituentStart = 0;
		for(int i = 0; i < wordList.size(); i++) {
			if(wordList.get(i).getLeftAnchor() == constituent.getLeftAnchor()) {
				constituentStart = i;
				break;
			}
		}
		int relativeStart = (int) ((int) constituentStart/(double)elements * 50D);
		startPosition.increment(relativeStart);
		int constituentEnd = 0;
		for(int i = 0; i < wordList.size(); i++) {
			if(wordList.get(i).getRightAnchor() == constituent.getRightAnchor()) {
				constituentEnd = i;
				break;
			}
		}
		int relativeEnd = (int) (constituentEnd / (double) elements * 50D);
		endPosition.increment(relativeEnd);
		constituentLength.increment(constituentEnd-constituentStart);
		int relativeConstituentLength = (int) ((int) (constituentEnd - constituentStart) / (double) elements * 100D);
		relativeConstituentsLengths.increment(relativeConstituentLength);
		return false;
	}
}
