package de.uni_koeln.spinfo.tesla.component.paradigms;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import de.uni_koeln.spinfo.tesla.annotation.adapter.IAnnotationFactory;
import de.uni_koeln.spinfo.tesla.annotation.adapter.IOutputAdapter;
import de.uni_koeln.spinfo.tesla.annotation.adapter.InputIterator;
import de.uni_koeln.spinfo.tesla.annotation.adapter.TypeMapping;
import de.uni_koeln.spinfo.tesla.annotation.adapter.tunguska.DefaultTunguskaOutputAdapter;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.access.IAlignedSequenceAccessAdapter;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.IAlignedSequence;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.GeneralizedConstituent;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.JustifiedHypothesis;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.Reason;
import de.uni_koeln.spinfo.tesla.component.util.MatrixLoader;
import de.uni_koeln.spinfo.tesla.roles.core.access.IAnchoredElementAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.roles.parser.impl.tunguska.access.TunguskaConstituentAccessAdapter.DefaultTunguskaConstituentAccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.Result;
import de.uni_koeln.spinfo.tesla.runtime.TeslaComponent;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.AccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Author;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.ChoiceRestriction;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Component;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Configuration;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Description;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.OutputAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.RoleDescription;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Run;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.ThreadMode;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;

@Component(threadMode=ThreadMode.CUSTOM, 
		author=@Author(	author="Stephan Schwiebert", 
						email="sschwieb@spinfo.uni-koeln.de", 
						web="http://www.spinfo.uni-koeln.de/space/sschwieb", 
						organization="Sprachliche Informationsverarbeitung"),
		description=@Description(name="SOG (Select)",
								 summary="",
								 bigO="l",
								 version="1.0",
								 reusableResults=true))
public class HypothesisEvaluator extends TeslaComponent {

	private static final long serialVersionUID = 1214407936563L;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.structure.AlignmentHypothesisGenerator", name = "Hypotheses", description = "The alignment hypotheses")
	private IAlignedSequenceAccessAdapter<IAlignedSequence> hypotheses;
	
	@OutputAdapter(dataObject=GeneralizedConstituent.class, type=DefaultTunguskaOutputAdapter.ProtoStuff.class, name="Constituents", 
			accessAdapterImpl=DefaultTunguskaConstituentAccessAdapter.class,
			description="All constituents detected by the parser")
	@RoleDescription("de.uni_koeln.spinfo.tesla.roles.structure.bootstrapping.GeneralizedHeuristicConstituentDetector")
	private IOutputAdapter<GeneralizedConstituent> out;
		
	@Configuration(name="Categorization", description="This parameter defines how categories will be defined to constituents. " +
			"'" + Categorizer.UNIQUE + "'" + " will generate a new category for each constituent. " +
			"'" + Categorizer.CONSTITUENT + "'" + " will assign the same category to two constituents if they refer to annotations of the same type. " +
			"'" + Categorizer.LEFT_CONTEXT + "'" + " will assign the same category to  all constituents which share the same left context.\n" +
			"'" + Categorizer.RIGHT_CONTEXT + "'" + " will assign the same category to all constituents which share the same right context.\n" +
			"'" + Categorizer.BOTH_CONTEXTS + "'" + " will assign the same category to all constituents which have the same left and right contexts in common. "
			, 
			editor="de.uni_koeln.spinfo.tesla.client.ui.editors.form.configurations.itemeditors.ChoiceEditor", 
			defaultValue = Categorizer.BOTH_CONTEXTS, restriction=ChoiceRestriction.class, editorConfiguration={Categorizer.BOTH_CONTEXTS, Categorizer.LEFT_CONTEXT, Categorizer.RIGHT_CONTEXT, Categorizer.CONSTITUENT, Categorizer.UNIQUE})
	private String categorizeMethod = Categorizer.BOTH_CONTEXTS;

	@Configuration(name = "Include sequence start", defaultValue = "true")
	private boolean alwaysAddStart = true;

	@Configuration(name = "Include sequence end", defaultValue = "true")
	private boolean alwaysAddEnd = true;
	
	@Configuration(name = "Sequence Start Weight", defaultValue = "1")
	private int sentenceStart = 1;

	@Configuration(name = "Sequence End Weight", defaultValue = "1")
	private int sentenceEnd = 1;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", name = "Sequences", description = "")
	private IAnchoredElementAccessAdapter<IAnchoredElement> sequences;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", name = "Sequence Items", description = "")
	private IAnchoredElementAccessAdapter<IAnchoredElement> words;

	private Categorizer categorizer;

	class ProcessCallable implements Callable<List<Annotation<GeneralizedConstituent>>> {
		
		private final GlobalAlignmentDetector detector;
		
		public ProcessCallable(GlobalAlignmentDetector detector) {
			this.detector = detector;
		}
		
		private Annotation<IAlignedSequence> sequence;

		public void setSequence(Annotation<IAlignedSequence> sequence) {
			this.sequence = sequence;
		}
	
		@Override
		public List<Annotation<GeneralizedConstituent>> call() throws Exception {
			Collection<JustifiedHypothesis> hypotheses = detector.analyzeSequence(sequence.getDataObject(), sentenceStart, sentenceEnd);
			Iterator<JustifiedHypothesis> iterator = hypotheses.iterator();
			List<Annotation<IAnchoredElement>> words = sequence.getDataObject().getWords();
			List<Annotation<GeneralizedConstituent>> toStore = new ArrayList<Annotation<GeneralizedConstituent>>();
			IAnnotationFactory factory = out.getAnnotationFactory();
			
			while(iterator.hasNext()) {
				JustifiedHypothesis constituent = iterator.next();
				GeneralizedConstituent data = new GeneralizedConstituent(constituent.getLeft(), constituent.getRight(), (int)constituent.getAverageLeft(), (int)constituent.getAverageRight(), (short) words.size());
				List<Reason> reasons = constituent.getReasons();
//				for (Reason r : reasons) {
//					
//				}
//				if(reasons.isEmpty()) {
//					System.out.println("ER!" );
//					continue;
//				}
				data.addContextDetails(reasons);
				int lastIncluded = constituent.getRight()-1;
				int left = constituent.getLeft();
				Annotation<GeneralizedConstituent> anno = factory.newAnnotation(words.get(constituent.getLeft()).getLeftAnchor(), words.get(lastIncluded).getRightAnchor(), words.get(0).getSignalId(), data, TypeMapping.SELF_GENERATED);
				categorizer.categorize(constituent, data, words, left, constituent.getRight(), (int)constituent.getAverageLeft(), (int)constituent.getAverageRight());
				toStore.add(anno);
				iterator.remove();
			}
			return toStore;
		}
		
	}
	
	@Run
	public Result run() throws Exception {
		categorizer = new Categorizer(categorizeMethod);
		InputIterator<Annotation<IAlignedSequence>> allSequences = hypotheses.getAlignedSequences();
		setProgressName("Loading data...");
		MatrixLoader ml = new MatrixLoader();
		InputIterator<String> signals = words.getAllSignalIds();
		List<String> signalIds = new ArrayList<String>();
		while(signals.hasNext()) {
			   signalIds.add(signals.next());
		}
		ArrayList<int[]> sequenceMatrix = ml.loadData(signalIds, this, sequences, words, 20);
		double step = 95D / sequenceMatrix.size();
		double current = 5;
		ArrayList<List<Annotation<IAnchoredElement>>> dataMatrix = ml.getDataMatrix();
		ml.release();
		setProgress((int) current);
		logger.info("Creating hypotheses...");
		GlobalAlignmentDetector detector = new GlobalAlignmentDetector(alwaysAddStart, alwaysAddEnd, sequenceMatrix, dataMatrix);
		int counter = 0;
		int threads = Math.max(2, Runtime.getRuntime().availableProcessors());
		ExecutorService executors = Executors.newFixedThreadPool(threads);
		List<ProcessCallable> pool = new ArrayList<ProcessCallable>(threads);
		for(int i = 0; i < threads; i++) {
			pool.add(new ProcessCallable(detector));
		}
		List<ProcessCallable> callables = new ArrayList<ProcessCallable>(threads);
		List<Future<List<Annotation<GeneralizedConstituent>>>> results = new ArrayList<Future<List<Annotation<GeneralizedConstituent>>>>();
		int next = 0;
		while(allSequences.hasNext() && !cancelled()) {
			
			
			Annotation<IAlignedSequence> sequence = allSequences.next();
			ProcessCallable callable = pool.get(next++);
			callable.setSequence(sequence);
			callables.add(callable);
			results.add(executors.submit(callable));
			if(next == threads) {
				next = store(callables, results);
			}
			
			counter++;
			if(counter % 1000 == 0) {
				logger.info("Analyzed " + counter + " sequences.");
			}
			current += step;
			setProgress((int) current);
		}
		executors.shutdown();
		store(callables, results);
		logger.info("Processed " + counter + " sequences.");
		return cancelled() ? Result.CANCELLED : Result.OK;
	}


	protected int store(List<ProcessCallable> callables,
			List<Future<List<Annotation<GeneralizedConstituent>>>> results)
			throws InterruptedException, ExecutionException {
		int next;
		for (Future<List<Annotation<GeneralizedConstituent>>> result : results) {
			/**
			 * Constituents are
			 * - guaranteed to be minimal: startword does not match any startword of the justifying constituents (same with endword)
			 * - guaranteed to be unique: no duplicates
			 * - guaranteed to be sorted: by left, than right anchor
			 * - define first and last included element (sublist-style)
			 * - refer to all "justifying" constituents
			 * 
			 */
			List<Annotation<GeneralizedConstituent>> toStore = result.get();
			for (Annotation<GeneralizedConstituent> anno : toStore) {
				out.store(anno);
			}
		}
		next = 0;
		callables.clear();
		results.clear();
		return next;
	}


	
	
}
