package de.uni_koeln.spinfo.tesla.component.test;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.junit.Assert;
import org.junit.Test;

import com.dyuproject.protostuff.GraphIOUtil;
import com.dyuproject.protostuff.LinkedBuffer;
import com.dyuproject.protostuff.Schema;
import com.dyuproject.protostuff.runtime.RuntimeSchema;

import de.uni_koeln.spinfo.tesla.component.alignhypothesis.HypothesisGenerator;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.IMatch;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.impl.AlignedSequence;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.INode;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.impl.NGramTree;
import de.uni_koeln.spinfo.tesla.component.paradigms.GlobalAlignmentDetector;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.IHypothesis;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.JustifiedHypothesis;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;




public class TestNaiveExhaustiveDetector {

	private Map<String, Integer> wordsToIds = new HashMap<String, Integer>();
	private Map<Integer, String> idsToWords = new HashMap<Integer, String>();
	
	public int[] getMapping(List<String> words) {
		int[] mapping = new int[words.size()];
		for(int i = 0; i < words.size(); i++) {
			String word = words.get(i);
			Integer id = wordsToIds.get(word);
			if(id == null) {
				id = wordsToIds.size();
				wordsToIds.put(word, id);
				idsToWords.put(id, word);
			}
			mapping[i] = id;
		}
		return mapping;
	}
	
	public List<String> getWordList(String sentence) {
		List<String> words = new ArrayList<String>();
		String[] splitted = sentence.split(" ");
		for (String word : splitted) {
			words.add(word);
		}
		return words;
	}
	
	@Test
	public void main() {
		List<List<String>> sentences = new ArrayList<List<String>>();
		// Generate sequences to insert
		// Paradigm on right side
		sentences.add(getWordList("Ich esse Fisch")); // 1
		sentences.add(getWordList("Ich esse Fleisch")); // 2
		// Paradigm on left side
		sentences.add(getWordList("Peter liebt Maria")); // 3
		sentences.add(getWordList("Hans liebt Maria")); // 4
//		// Paradigm within
		sentences.add(getWordList("Er kauft Fisch")); // 5
		sentences.add(getWordList("Er hasst Fisch")); // 6
//		// Paradigm on right side/null
		sentences.add(getWordList("Anton mag Petra")); // 7
		sentences.add(getWordList("Anton mag Petra sehr")); // 8
////		// Paradigm on left side, different length
		sentences.add(getWordList("Herr Meier tanzt besessen vor sich hin")); // 9
		sentences.add(getWordList("Karl tanzt besessen vor sich hin")); // 10
//		// Paradigm within, different length
		sentences.add(getWordList("Die alte Frau trinkt eklige warme Cola")); // 11
		sentences.add(getWordList("Die alte Frau trinkt kalte Cola")); // 12
//		// Multi align, different sequences:
		sentences.add(getWordList("Ich esse sehr gerne Fisch")); // 13
//		// Paradigm on left side/null
		sentences.add(getWordList("Michael Jackson lebt")); // 14;
		sentences.add(getWordList("Jackson lebt")); // 15
		// Paradigm on right side/null
		sentences.add(getWordList("Das stimmt")); // 16
		sentences.add(getWordList("Das stimmt nicht")); // 17
		
		sentences.add(getWordList("Petra liebt Anton")); // 18
		// Paradigm due to sequence end only
		sentences.add(getWordList("Ein Vogel fliegt")); // 19
		sentences.add(getWordList("Ein Frosch schwimmt")); // 20
		// Paradigm due to sequence start only
		sentences.add(getWordList("Otto gibt auf")); // 21
		sentences.add(getWordList("Anna steht auf")); // 22
		
		
		
		
		/*
		 * Expectation:
		 * 
		 * "Ich esse" (1 and 5/6)
		 * "Fisch" (1 and 2)
		 * "Fleisch" (2 and 1)
		 * "Peter" ( 3 and 4)
		 * "Hans" (4 and 3)
		 * "kauft" (5 and 6)
		 * "kauft Fisch" (5 and 6)
		 * "Er kauft" (5 and 1)
		 * "Er hasst" (6 and 1)
		 * "hasst" (6 and 5)
		 * "hasst Fisch" (6 and 5)
		 * "sehr" (7 and 8)
		 * "Herr Meier" (9 and 10)
		 * "Karl" (10 and 9)
		 * "eklige warme" (11 and 12)
		 * "kalte" (12 and 11)
		 * "eklige warme Cola" (11 and 12)
		 * "kalte Cola" (12 and 11)
		 * "Die alte Frau trinkt eklige warme" (11 and 12)
		 * "Die alte Frau trinkt kalte" (12 and 11)
		 * "Ich esse sehr gerne" (13 and 1)
		 * "sehr gerne" (13 and 1)
		 * "gerne Fisch" (13 and 8)
		 * "sehr gerne Fisch" (1 and 13)
		 * "Anton mag Petra" (8 and 13)
		 * "Michael" (14 and 15)
		 * "nicht" (16 and 17)
		 * "Petra" (18 and (3,4))
		 * "Anton"
		 * "Maria"
		 * unwanted but still expected:
		 * "mag Petra" (18 and 7)
		 * "mag Petra sehr" (18 and 8)
		 * "Anton mag" (7 and 18, 8 and 18, twice)
		 * "Petra liebt"
		 * 
		 * plus all sentences
		 * 
		 */
		ArrayList<int[]> sequences = new ArrayList<int[]>();
		Set<List<String>> expected = new HashSet<List<String>>();
	//	expected.addAll(sentences);
		expected.add(getWordList("Ich esse"));
		expected.add(getWordList("Fisch"));
		expected.add(getWordList("Fleisch"));
		expected.add(getWordList("Peter"));
		expected.add(getWordList("Hans"));
		expected.add(getWordList("kauft"));
		expected.add(getWordList("kauft Fisch"));
		expected.add(getWordList("Er kauft"));
		expected.add(getWordList("Er hasst"));
		expected.add(getWordList("hasst"));
		expected.add(getWordList("hasst Fisch"));
		expected.add(getWordList("sehr"));
		expected.add(getWordList("Herr Meier"));
		expected.add(getWordList("Karl"));
		expected.add(getWordList("eklige warme"));
		expected.add(getWordList("kalte"));
		expected.add(getWordList("eklige warme Cola"));
		expected.add(getWordList("kalte Cola"));
		expected.add(getWordList("Die alte Frau trinkt kalte"));
		expected.add(getWordList("Die alte Frau trinkt eklige warme"));
		expected.add(getWordList("Ich esse sehr gerne"));
		expected.add(getWordList("sehr gerne"));
		expected.add(getWordList("gerne Fisch"));
		expected.add(getWordList("sehr gerne Fisch"));
		expected.add(getWordList("Anton mag Petra"));
		expected.add(getWordList("Michael"));
		expected.add(getWordList("nicht"));
		expected.add(getWordList("Petra"));
		expected.add(getWordList("Anton"));
		expected.add(getWordList("Petra liebt"));
		expected.add(getWordList("Maria"));
		expected.add(getWordList("mag Petra sehr"));
		expected.add(getWordList("Anton mag"));
		expected.add(getWordList("mag Petra"));
		expected.add(getWordList("liebt Anton"));
		expected.add(getWordList("Vogel fliegt"));
		expected.add(getWordList("Frosch schwimmt"));
		expected.add(getWordList("Otto gibt"));
		expected.add(getWordList("Anna steht"));
		for (List<String> sentence : sentences) {
			int[] mapping = getMapping(sentence);
			sequences.add(mapping);
		}
		Set<List<String>> expectedTwice = new HashSet<List<String>>();
		expectedTwice.add(getWordList("Ich esse"));
		expectedTwice.add(getWordList("Maria"));
		expectedTwice.add(getWordList("Anton mag"));

		/**
		 * Kleiner Denkfehler bei den belegen:
		 * a) Peter liebt Maria
		 * b) Petra liebt Anton
		 * 
		 * daraus folgt u.a., dass Maria/Anton eine Struktur sind (völlig ok)
		 * aber:
		 * 
		 * c) Hans liebt Maria
		 * hier wird "liebt" ebenfalls als Strukturanfang markiert,
		 * so dass für "liebt Maria" in a) auch "liebt Maria" aus c)
		 * als "Beleg" genommen wird.
		 * 
		 * Damit eine hypothese also "wirklich richtig" ist, muss
		 * sichergestellt werden, dass sich die ersten/letzten Wörter
		 * von Hypothese und Beleg unterscheiden!!
		 * 
		 * Beim Einfügen von möglichen Start- und End-Positionen
		 * in die AlignedSequence sollte also unterschieden bzw. berücksichtigt
		 * werden, wie der Inhalt der Struktur in beiden
		 * Hypothesen weitergeht... das können theoretisch beliebig
		 * viele Wörter sein, einschließlich der kompletten Hypothese.
		 * 
		 *  Knoten-Vergleich? 
		 *  
		 */
		
		NGramTree suffixes = new NGramTree(200);
		int counter = 0;
		for (int[] sentence : sequences) {
			suffixes.insert(sentence, counter, false);
			counter++;
		}
		suffixes.trim();
		
		NGramTree prefixes = new NGramTree(200);
		counter = 0;
		for (int[] sentence : sequences) {
			prefixes.insert(sentence, counter, true);
			counter++;
		}
		prefixes.reversePositions(sequences);
		
		final List<AlignedSequence> sList = new ArrayList<AlignedSequence>();
		HypothesisGenerator align = new HypothesisGenerator() {
			
			@Override
			protected void storeResults(List<Future<AlignedSequence>> results,
					ArrayList<List<Annotation<IAnchoredElement>>> dataMatrix,
					ArrayList<Annotation<IAnchoredElement>> baseAnnotations)
					throws InterruptedException, ExecutionException {
				Iterator<Future<AlignedSequence>> iterator = results.iterator();
				while(iterator.hasNext()) {
					AlignedSequence alignedSequence = iterator.next().get();
					sList.add(alignedSequence);
				}
			}
			
		};
		//align.setMatrix(sequences);
		final Map<Integer, List<INode>> suffixNodes = align.getTypeMapping(1, 1, suffixes);
		final Map<Integer, List<INode>> prefixNodes = align.getTypeMapping(1, 1, prefixes);
		//detector.setSequences(sequences);
		
		
		align.setSequences(sequences);
		try {
			align.process(suffixNodes, prefixNodes);
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (ExecutionException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		final List<JustifiedHypothesis> all = new ArrayList<JustifiedHypothesis>();
		GlobalAlignmentDetector detector = new GlobalAlignmentDetector(true, true, sequences, null);
		for (int i = 0; i < sList.size(); i++) {
			final AlignedSequence sequence = sList.get(i);
			Collection<JustifiedHypothesis> hypotheses = detector.analyzeSequence(sequence, 0, 0);
			all.addAll(hypotheses);
			
			if(i % 1000 == 0) {
				System.out.println("Processed " + i + " sequences.");
			}
			
		}
		
		for(IHypothesis constituent : all) {
			int[] list = sequences.get(constituent.getSequence());
			List<String> original = convertBack(list);
			System.out.println("Sentence " + constituent.getSequence() + ": " + original);
			List<String> consti = original.subList(constituent.getLeft(), constituent.getRight());
			System.out.println("       Constituent: " + consti);
			TestReasonList reasons = (TestReasonList) constituent.getReasons();
			for (IMatch match : reasons.getMatches()) {
				List<String> other = convertBack(sequences.get(match.sequence()));
				//int left = Math.max(0,(match.start.wordPos() - match.start.getContextLength()));
				//int right = match.start.wordPos();
				//List<String> leftList = other.subList(left, right);
				System.out.println("LC: " + match.start().wordPos() + " in " + other);
				System.out.println("RC: " + match.end().wordPos() + " in " + other);
				//System.out.println("         Context: " + leftList + "    XXXX    " + " in " + other);
			}
			boolean removed = expected.remove(consti);
			if(!removed) {
				removed = expectedTwice.remove(consti);
			}
			if(removed) {
				System.out.println("SUCCESS!!!");
			}
			if(!removed) {
				System.out.println("FAILURE: Didn't find or expect " + consti + "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!");
			}
			System.out.println();
			System.out.println();
		}
		if(!expected.isEmpty()) {
			System.out.println("Missed first time: " + expected);
		}
		Assert.assertTrue(expected.isEmpty());
		if(!expectedTwice.isEmpty()) {
			System.out.println("Missed second time: " + expectedTwice);
		}
		Assert.assertTrue(expectedTwice.isEmpty());
		try {
			testStoreAndLoad(all);
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
	
	
	class TestContainer {
		
		private List<JustifiedHypothesis> list;
		
	}
	
	
	private void testStoreAndLoad(List<JustifiedHypothesis> all) throws IOException {
		File file = File.createTempFile("test", "test");
		file.deleteOnExit();
		TestContainer test = new TestContainer();
		test.list = new ArrayList<JustifiedHypothesis>(all);
		DataOutputStream out = new DataOutputStream(new FileOutputStream(file));
		ArrayList<JustifiedHypothesis> backup = new ArrayList<JustifiedHypothesis>(all);
		Schema schema = RuntimeSchema.createFrom(test.getClass());
		byte[] bytes = toBytes(test, schema);
		System.out.println("Writing: " + bytes.length);
		out.writeInt(bytes.length);
		out.write(bytes, 0, bytes.length);
		out.close();
		DataInputStream in = new DataInputStream(new FileInputStream(file));
		TestContainer test2 =new TestContainer();
		int size = in.readInt();
		byte[] data = new byte[size];
		in.readFully(data);
		fromBytes(data, schema, test2, size);
		List<JustifiedHypothesis> listA = test.list;
		List<JustifiedHypothesis> listB = test2.list;
		for(int i = 0; i < listA.size(); i++) {
			if(listA.get(i).getLeft() != listB.get(i).getLeft()) {
				System.out.println("Falsch!");
			}
		}
		
	}
	
	
private static LinkedBuffer buffer = LinkedBuffer.allocate(256);
	
	protected static byte[] toBytes(Object dataObject, Schema schema) {
		
		byte[] array = GraphIOUtil.toByteArray(dataObject, schema, buffer);
		buffer.clear();
		return array;
	}
	
	protected static <T> void fromBytes(byte[] data, Schema<T> schema, T object, int length) {
		GraphIOUtil.mergeFrom(data, 0, length, object, schema);
	}
	
	private List<String> convertBack(int[] list) {
		List<String> original = new ArrayList<String>(list.length);
		for (Integer value : list) {
			original.add(idsToWords.get(value));
		}
		return original;
	}

	
}
