package de.uni_koeln.spinfo.tesla.component.alignhypothesis;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.log4j.Logger;

import de.uni_koeln.spinfo.tesla.annotation.adapter.IOutputAdapter;
import de.uni_koeln.spinfo.tesla.annotation.adapter.InputIterator;
import de.uni_koeln.spinfo.tesla.annotation.adapter.TypeMapping;
import de.uni_koeln.spinfo.tesla.annotation.adapter.tunguska.DefaultTunguskaOutputAdapter;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.access.impl.AlignedSequenceAccessAdapter;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.impl.AlignedSequence;
import de.uni_koeln.spinfo.tesla.component.ngramtree.access.INgramTreeAccessAdapter;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.INgramTree;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.INode;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.IPosition;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.impl.Node;
import de.uni_koeln.spinfo.tesla.component.util.MatrixLoader;
import de.uni_koeln.spinfo.tesla.roles.core.access.IAnchoredElementAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.runtime.Result;
import de.uni_koeln.spinfo.tesla.runtime.TeslaComponent;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.AccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Author;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Component;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Configuration;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Description;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.OutputAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.RoleDescription;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Run;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.ThreadMode;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;

@Component(threadMode=ThreadMode.CUSTOM, 
		author=@Author(	author="Stephan Schwiebert", 
						email="sschwieb@spinfo.uni-koeln.de", 
						web="http://www.spinfo.uni-koeln.de/space/sschwieb", 
						organization="Sprachliche Informationsverarbeitung"),
		description=@Description(name="SOG (Boundaries)",
								 summary="Generates structural hypotheses based on the shared context of two or more sequences of elements.",
								 bigO="",
								 version="1.0",
								 reusableResults=true))
public class HypothesisGenerator extends TeslaComponent {
	
	private static final long serialVersionUID = 1214407936563L;

	@Configuration(name = "Minimum Context Width Left", defaultValue = "2")
	private int minLeft = 2;

	@Configuration(name = "Minimum Context Width Right", defaultValue = "2")
	private int minRight = 2;

	@Configuration(name = "Sequence Start Weight", defaultValue = "1")
	private int sentenceStart = 1;

	@Configuration(name = "Sequence End Weight", defaultValue = "1")
	private int sentenceEnd = 1;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.structure.ngramTreeGenerator", name = "Suffix Tree")
	private INgramTreeAccessAdapter<INgramTree> suffixTree;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.structure.ngramTreeGenerator", name = "Prefix Tree")
	private INgramTreeAccessAdapter<INgramTree> prefixTree;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", name = "Sequences", description = "The sequences to align (for instance, sentences)")
	private IAnchoredElementAccessAdapter<IAnchoredElement> sequences;

	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", name = "Sequence Items", description = "The items to align (for instance, words)")
	private IAnchoredElementAccessAdapter<IAnchoredElement> words;

	@OutputAdapter(dataObject = AlignedSequence.class, type = DefaultTunguskaOutputAdapter.class, name = "Hypotheses", accessAdapterImpl = AlignedSequenceAccessAdapter.class, description = "Alignment Hypotheses", bufferSize = 50)
	@RoleDescription("de.uni_koeln.spinfo.tesla.roles.structure.AlignmentHypothesisGenerator")
	private IOutputAdapter<AlignedSequence> out;

	private int maxThreads = 2;

	private ArrayList<List<Annotation<IAnchoredElement>>> dataMatrix;

	private ArrayList<Annotation<IAnchoredElement>> sequencesList;
	
	private ExecutorService executors = Executors.newFixedThreadPool(maxThreads);
		
	private Logger logger = Logger.getLogger(getClass());

	private ArrayList<int[]> sequenceMatrix;

	
	@Run
	public Result run() throws Exception {
		setProgressName("Preprocessing");
		logger.info("Loading data...");
		MatrixLoader loader = new MatrixLoader();
		InputIterator<String> signals = words.getAllSignalIds();
		ArrayList<String> signalIds = new ArrayList<String>();
		while(signals.hasNext()) {
			signalIds.add(signals.next());
		}
		
		sequenceMatrix = loader.loadData(signalIds, this, sequences, words, 20D);
		dataMatrix = loader.getDataMatrix();
		sequencesList = loader.getSequencesList();
		loader.release();
		if (cancelled()) {
			return Result.CANCELLED;
		}
		setProgressName("Preprocessing");
		maxThreads = (int) (Math.max(2, Runtime.getRuntime().availableProcessors()/2));
		executors = Executors.newFixedThreadPool(maxThreads);
		logger.info("Optimizing N-Gram trees");
		final Map<Integer, List<INode>> suffixNodes =  getTypeMapping(suffixTree, minLeft, sentenceStart);
		final Map<Integer, List<INode>> prefixNodes = getTypeMapping(prefixTree, minRight, sentenceEnd);
		process(suffixNodes, prefixNodes);
		sequenceMatrix.clear();
		if (cancelled()) {
			return Result.CANCELLED;
		}
		super.setProgressName("Start- and endpoints of all hypotheses were generated.");
		return Result.OK;
	}
	

	
	
	public void process(Map<Integer, List<INode>> suffixNodes, Map<Integer, List<INode>> prefixNodes)
			throws InterruptedException, ExecutionException {
		
		List<Future<AlignedSequence>> results = new ArrayList<Future<AlignedSequence>>();
		List<BoundsMarker> pool = new ArrayList<BoundsMarker>(maxThreads);
		for(int i = 0; i < maxThreads; i++) {
			BoundsMarker processor = new BoundsMarker();
			processor.setSequenceMatrix(sequenceMatrix);
			processor.setSuffixNodes(suffixNodes);
			processor.setPrefixNodes(prefixNodes);
			pool.add(processor);
		}
		List<BoundsMarker> callables = new ArrayList<BoundsMarker>();
		double current = 20D;
		double step = 80D / sequenceMatrix.size();
		int next = 0;
		for (int i = 0; i < sequenceMatrix.size(); i++) {
			BoundsMarker callable = pool.get(next++);
			callable.setSequenceId(i);
			callables.add(callable);
			results.add(executors.submit(callable));
			if(next == maxThreads) {
				storeResults(results, dataMatrix, sequencesList);
				callables.clear();
				results.clear();
				next = 0;
			}
			if(i % 2500 == 0) {
				logger.info("Processed " + i + " sequences.");
			}
			current += step;
			setProgress((int) current);
			if(cancelled()) break;
			
		}
		logger.info("Storing last results...");
		executors.shutdown();
		storeResults(results, dataMatrix, sequencesList);
		logger.info("Releasing resources...");
		
	}


	protected void storeResults(
			List<Future<AlignedSequence>> results, ArrayList<List<Annotation<IAnchoredElement>>> dataMatrix, ArrayList<Annotation<IAnchoredElement>> baseAnnotations)
			throws InterruptedException, ExecutionException {
		
		for(int x = 0; x < results.size(); x++) {
			AlignedSequence constituents = results.get(x).get();
			if(constituents == null) {
				continue;
			}
			constituents.setSequence(dataMatrix.get(constituents.getSequenceId()));
			Annotation<AlignedSequence> anno = out.getAnnotationFactory().newAnnotation(baseAnnotations.get(constituents.getSequenceId()), constituents, TypeMapping.NONE);
			out.store(anno);
		}
		results.clear();
	}

	

	
	private Map<Integer, List<INode>> getTypeMapping(
			INgramTreeAccessAdapter<INgramTree> adapter, int minDepth, int bonus) {

		INgramTree tree = adapter.getTree();
		Map<Integer, List<INode>> nodesByType = getTypeMapping(minDepth, bonus, tree);
		adapter.release();
		adapter.close();
		return nodesByType;
	}

	

	public Map<Integer, List<INode>> getTypeMapping(int minDepth, int bonus,
			INgramTree tree) {
		logger.info("Preparing type to node mapping");
		Iterator<INode> nodes = tree.iterateAndRelease();
		Map<Integer, List<INode>> nodesByType = new HashMap<Integer, List<INode>>();
		while (nodes.hasNext()) {
			INode node = nodes.next();
			if(node.getDepth() <= 0) continue;
			if(!node.isBranching()) continue;
			List<INode> list = nodesByType.get(node.getValue());
			if (list == null) {
				list = new ArrayList<INode>();
				nodesByType.put(node.getValue(), list);
			}
			if (node.getDepth() < minDepth) {
				int depth = node.getDepth();
				if (depth + bonus >= minDepth) {
					Node newNode = new Node(node.getValue(), node.getDepth());
					newNode.setBranching(true);
					int requiredDepth = node.getDepth()-1;
					Collection<IPosition> referencedPositions = node.getReferencedPositions();
					for (IPosition pp : referencedPositions) {
						if(pp.getElementIndex() == requiredDepth ) {
							newNode.addPosition(pp);
						}
					}
					list.add(newNode);
				}
				continue;
			}
			Node copy = new Node(node.getValue(), node.getDepth());
			copy.setPositions(node.getReferencedPositions());
			copy.setBranching(true);
			list.add(copy);
		}
		tree = null;
		return nodesByType;
	}
	
	@Override
	public boolean cancelled() {
		return super.cancelled();
	}




	public void setSequences(ArrayList<int[]> sequences) {
		this.sequenceMatrix = sequences;
	}


}
