package de.uni_koeln.spinfo.tesla.component.ngramtree.data.impl;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import de.uni_koeln.spinfo.tesla.component.ngramtree.data.INgramTree;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.INode;
import de.uni_koeln.spinfo.tesla.component.ngramtree.data.IPosition;
import de.uni_koeln.spinfo.tesla.component.util.GoogleHashSet;



/**
 * This is not a suffix tree, but an N-Gram-Tree, designed to store words of a sentence.
 * It is created naive, which means that the execution time mainly depends on
 * the overall number of words multiplied by N. Assuming that sentences in natural languages
 * are <i>not very long</i>, this is acceptable, especially as the nodes in the
 * tree store additional sentence information: Each node represents a word,
 * and internally stores information about the sentence it occurred in, and the
 * word position within this sentence. Therefore, mapping a node back to a word
 * in a sentence can be done in constant time.
 * 
 * @author sschwieb
 *
 */
public class NGramTree implements INgramTree {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = -6595784693922413177L;

	/**
	 * Termination symbol.
	 */
	public static final int TERMINATE = -2;
	
	protected final Node root = new Node(-1,0);
		
	private int maxDepth;

	private long id;
	
	public NGramTree() {}
	
	public NGramTree(int maxDepth) {
		this.maxDepth = maxDepth;
	}

	public Node getRoot() {
		return root;
	}

	public void insert(final int[] sequence, final int id, final boolean reverse) {
		final Node searchDummy = new Node(Integer.MAX_VALUE, 0);
		int[] path = sequence;
		if(reverse) {
			path = flip(path);
		}
		for(int i = 0; i < path.length; i++) {
			Node current = root;
			int steps = 0;
			for(int j = i; j < path.length; j++) {
				current = current.append(path[j], id, (short) j, searchDummy, this);
				steps++;
				if(steps == maxDepth) break;
			}
			current.append(TERMINATE, id, (short) 255, searchDummy, null);
		}	
	}
	
	
	private int[] flip(int[] path) {
		int[] reverse = new int[path.length];
		for(int i = 0; i < reverse.length; i++) {
			reverse[i] = path[path.length-i-1];
		}
		return reverse;
	}


	public void printTree() {
		print(root);
		System.out.println();
	}


	private void print(INode current) {
		System.out.println(current.getValue() + ", Depth " + current.getDepth());
		Set<INode> children = current.getChildren();
		if(children == null) return;
		for (INode node : children) {
			print(node);
		}
	}

	public Iterator<INode> iterateAndRelease() {
		return new NodeIterator(root);
	}

	class NodeIterator implements Iterator<INode> {
		
		private Node toReturn;

		public NodeIterator(Node root) {
			findNext(root);
		}
		
		public void findNext(Node current) {
			if(current == null) {
				toReturn = null;
				return;
			}
			Node next = current;
			while(next.getChildren() != null && next.getChildren().size() > 0) {
				next = (Node) next.getChildren().iterator().next();
			}
			this.toReturn = next;
			toReturn.updateBranching();
		}
		
		public boolean hasNext() {
			return toReturn != null;
		}
		
		public Node next() {
			Node node = toReturn;
			toReturn.clearChildren();
			Node parent = toReturn.getParent();
			if(parent != null) {
				parent.updateBranching();
				parent.getChildren().remove(toReturn);
			}
			findNext(parent);
			return node;
		}

		@Override
		public void remove() {
			// TODO Auto-generated method stub
			
		}
		
		
		
	}

	@Override
	public long getId() {
		return id;
	}

	@Override
	public void setId(long id) {
		this.id = id;
	}

	public int getNumberOfNodes() {
		return count(root);
	}

	private int count(Node node) {
		if(node.getChildren() == null) return 1;
		int subTree = 1;
		GoogleHashSet<INode> children = node.getChildren();
		for (INode iNode : children) {
			subTree += count((Node) iNode);
		}
		return subTree;
	}

	public void reversePositions(ArrayList<int[]> sequenceMatrix) {
		reverse(root, sequenceMatrix);
	}
	
	
	private void reverse(Node current, ArrayList<int[]> sequenceMatrix) {
		GoogleHashSet<INode> children = current.getChildren();
		if(children != null) {
			for (INode child : children) {
				reverse((Node) child, sequenceMatrix);
			}
		}
		List<IPosition> reverse = reverse(current.getReferencedPositions(), sequenceMatrix);
		current.setPositions(reverse);
	}

	private List<IPosition> reverse(Collection<IPosition> positions, ArrayList<int[]> sequenceMatrix) {
		ArrayList<IPosition> reverse = new ArrayList<IPosition>(positions.size());
		for (IPosition posPair : positions) {
			PosPair rev = PosPair.newPair(posPair.getSequenceIndex(), (short) (sequenceMatrix.get(posPair.getSequenceIndex()).length - posPair.getElementIndex()-1));
			reverse.add(rev);
		}
		reverse.trimToSize();
		return reverse;
		
	}

	public void trim() {
		trim(root);
	}

	private void trim(Node current) {
		GoogleHashSet<INode> children = current.getChildren();
		if(children != null) {
			for (INode child : children) {
				trim((Node) child);
			}
		}
		List<IPosition> trimmed = new ArrayList<IPosition>(current.getReferencedPositions());
		current.setPositions(trimmed);
	}

}
