package de.uni_koeln.spinfo.tesla.component.paradigms;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.IAlignedSequence;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.IMatch;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.impl.AlignedSequence;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.impl.ContextPosition;
import de.uni_koeln.spinfo.tesla.component.alignhypothesis.data.impl.Match;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.JustifiedHypothesis;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.Reason;
import de.uni_koeln.spinfo.tesla.component.test.TestReasonList;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;


/**
 * 
 * @author sschwieb
 *
 */
public class GlobalAlignmentDetector {
	
	private final boolean alwaysAddHypothesisToSequenceEnd, alwaysAddHypothesisFromSequenceStart;
			
	private boolean excludeEmpty = true;
	private boolean excludeEmptyReason = false;

	private ArrayList<List<Annotation<IAnchoredElement>>> dataMatrix;

	private ArrayList<int[]> sequenceMatrix;
		
	public GlobalAlignmentDetector(boolean alwaysAddStart, boolean alwaysAddEnd, ArrayList<int[]> sequenceMatrix, ArrayList<List<Annotation<IAnchoredElement>>> dataMatrix ) {
		this.alwaysAddHypothesisToSequenceEnd = alwaysAddStart;
		this.alwaysAddHypothesisFromSequenceStart = alwaysAddEnd;
		this.dataMatrix = dataMatrix;
		this.sequenceMatrix = sequenceMatrix;
	}
	
	public List<Reason> getReasons(List<IMatch> matches, int forbiddenStartType, int forbiddenEndType, int startBonus, int endBonus) {
		ArrayList<Reason> reasons = new ArrayList<Reason>(256);
		if(dataMatrix == null) {
			System.out.println("Assuming debug mode - returning patched list (fix this!)");
			return new TestReasonList(matches);
		}
		for (IMatch match : matches) {
			List<Annotation<IAnchoredElement>> sequence = dataMatrix.get(match.sequence());
			ContextPosition start = match.start();
			ContextPosition end = match.end();
			try {
				if(start.wordPos() == sequence.size()-1) {
					// Adjunction at the end of the sentence: Add an empty element
					Annotation<IAnchoredElement> anchor = sequence.get(end.wordPos()-1);
					Reason r = new Reason();
					r.setSequence(match.sequence());
					r.setFrom(anchor.getRightAnchor());
					r.setTo(anchor.getRightAnchor());
					r.setLeftContextWidth(start.getContextLength());
					r.setRightContextWidth(end.getContextLength());
					updateStartEndBonus(startBonus, endBonus, start, end, r);
					reasons.add(r);
					
				} else if(end.wordPos() == 0) {
					// Adjunction at the beginning of the sentence: Add an empty element
					Annotation<IAnchoredElement> anchor = sequence.get(0);
					Reason r = new Reason();
					r.setSequence(match.sequence());
					r.setFrom(anchor.getLeftAnchor());
					r.setTo(anchor.getLeftAnchor());
					r.setLeftContextWidth(start.getContextLength());
					r.setRightContextWidth(end.getContextLength());
					updateStartEndBonus(startBonus, endBonus, start, end, r);
					reasons.add(r);
				} else {
					Annotation<IAnchoredElement> leftAnchor = sequence.get(start.wordPos()+1);
					Annotation<IAnchoredElement> rightAnchor = sequence.get(end.wordPos()-1);
					if(leftAnchor.getLeftAnchor() >= rightAnchor.getRightAnchor() && excludeEmptyReason) {
						// TODO: Empty hypothesis
					} else {
						Reason r = new Reason();
						r.setSequence(match.sequence());
						r.setFrom(leftAnchor.getLeftAnchor());
						r.setTo(rightAnchor.getRightAnchor());
						r.setLeftContextWidth(start.getContextLength());
						r.setRightContextWidth(end.getContextLength());
						updateStartEndBonus(startBonus, endBonus, start, end, r);
						// TODO: Add option
						if(accept(match.start(), match.end(), forbiddenStartType, forbiddenEndType))
							reasons.add(r);						
					}	
				}
				
			} catch (IndexOutOfBoundsException e) {
				// TODO: FIX THIS!
				e.printStackTrace();
			}
		}
		reasons.trimToSize();
		return reasons;
	}

	protected void updateStartEndBonus(int startBonus, int endBonus,
			ContextPosition start, ContextPosition end, Reason r) {
		/*
		 * Kontextlänge links: Maximal so, dass (start.wordPos() + 1) - length == 0.
		 */
		if(startBonus > 0) {
			if((start.wordPos() + 1) - start.getContextLength() <= 0) {
				r.setLeftBonus(startBonus);
			}
		}
		/*
		 * Kontextlänge rechts: Maximal so, dass (end.wordPos() - 1) + length == sequence length
		 */
		if(endBonus > 0) {
			int pos = end.wordPos()-1 + end.getContextLength();
			int max = sequenceMatrix.get(end.sequence()).length-1;
			if(pos >= max) {
				r.setRightBonus(endBonus);
			}
		}
	}

	private boolean accept(ContextPosition start, ContextPosition end, int forbiddenStartType, int forbiddenEndType) {
		int startType = sequenceMatrix.get(start.sequence())[start.wordPos()+1];
		int endType = sequenceMatrix.get(end.sequence())[end.wordPos()-1];
		if(startType == forbiddenStartType || endType == forbiddenEndType) {
			return false;
		}
		return true;
	}
	
	public Collection<JustifiedHypothesis> analyzeSequence(
			IAlignedSequence sequence, int startBonus, int endBonus) {
		SortedSet<Short> startIndices = sequence.getStartIndices();
		SortedSet<Short> endIndices = sequence.getEndIndices();
		Set<JustifiedHypothesis> hypotheses = new TreeSet<JustifiedHypothesis>(new Comparator<JustifiedHypothesis>() {

			@Override
			public int compare(JustifiedHypothesis o1, JustifiedHypothesis o2) {
				int start = o1.getLeft() - o2.getLeft();
				if(start == 0) {
					return o1.getRight() - o2.getRight();
				}
				return start;
			}
		});

		for (Short start : startIndices) {
			SortedSet<Short> endings = endIndices.tailSet(start);
			for (Short end : endings) {
				// Start and end are two positions in the current sequence and mark
				// the similar elements before and after the "constituent".
				// Thus, the constituent starts at "start+1" and does not include
				// the element at "end": 
				
				// end == start results in an illegal subsequence
				if(end == start) continue;
				// start == (end-1) is an empty subsequence
				if(excludeEmpty  && start == end - 1) continue;
				
				// Get all justifications for the current hypothesis
				List<IMatch> matches = sequence.getMatches(start, end);
				if(matches.size() == 0) {
					// start to end could have been a valid hypothesis, but no
					// justification was found - skip this hypothesis
					continue;
					
				}
				// JustifiedHypothesis start: index of the first word which belongs to the hypothesis
				// JustifiedHypothesis end: index of the first word which does NOT belong to the hypothesis
				// -> Default behaviour, as in sublist etc
				JustifiedHypothesis hypothesis = new JustifiedHypothesis(sequence.getSequenceId(), (short)(start+1), (short)(end));
				
				int forbiddenStart = sequenceMatrix.get(sequence.getSequenceId())[start+1];
				int forbiddenEnd = sequenceMatrix.get(sequence.getSequenceId())[end-1];
				
				List<Reason> reasons = getReasons(matches, forbiddenStart, forbiddenEnd, startBonus, endBonus);
				if(reasons.isEmpty()) {
					continue;
				}
				hypothesis.setReasons(reasons);
				hypotheses.add(hypothesis);
				
			}
			if(alwaysAddHypothesisToSequenceEnd) {
				if(excludeEmpty && start + 1 == sequence.getLength()) break;
				List<IMatch> matches = sequence.getSingleStartMatches(start,sequenceMatrix);
				if(matches.isEmpty()) continue;
				int forbiddenStart = sequenceMatrix.get(sequence.getSequenceId())[start+1];
				int[] seq = sequenceMatrix.get(sequence.getSequenceId());
				int forbiddenEnd = seq[seq.length-1];
				List<Reason> reasons = getReasons(matches, forbiddenStart, forbiddenEnd, startBonus, endBonus);
				if(reasons.isEmpty()) continue;
				JustifiedHypothesis hypothesis = new JustifiedHypothesis(sequence.getSequenceId(), (start+1), sequence.getLength());
				hypothesis.setReasons(reasons);
				hypotheses.add(hypothesis);
			}
		}
		if(alwaysAddHypothesisFromSequenceStart) {
			for (Short end : endIndices) {
				if(excludeEmpty && end == 0) continue;
				List<IMatch> matches = sequence.getSingleEndMatches(end, sequenceMatrix);
				if(matches.isEmpty()) continue;
				int forbiddenStart = sequenceMatrix.get(sequence.getSequenceId())[0];
				int forbiddenEnd = sequenceMatrix.get(sequence.getSequenceId())[end-1];
				List<Reason> reasons = getReasons(matches, forbiddenStart, forbiddenEnd, startBonus, endBonus);
				if(reasons.isEmpty()) continue;
				JustifiedHypothesis hypothesis = new JustifiedHypothesis(sequence.getSequenceId(), (short)0, (short)(end));
				hypothesis.setReasons(reasons);
				hypotheses.add(hypothesis);
			}
		}
		if(sequence instanceof AlignedSequence) {
			((AlignedSequence) sequence).release();
		}
		//System.out.println("Hypos:" + hypotheses.size());
		return hypotheses;
	}
	
	class MultiHypothesis {

		private TreeMap<Integer, TreeSet<Match>> matches = new TreeMap<Integer, TreeSet<Match>>();

		public void add(Match match) {
			TreeSet<Match> set = matches.get(match.sequence());
			if(set == null) {
				set = new TreeSet<Match>();
				matches.put(match.sequence(), set);
			}
			set.add(match);
		}

		public boolean foundMatches() {
			return matches.size() > 0;
		}
		
	}
	
	
	
}
