package de.uni_koeln.spinfo.tesla.component.paradigms.select;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import de.uni_koeln.spinfo.tesla.annotation.adapter.IOutputAdapter;
import de.uni_koeln.spinfo.tesla.annotation.adapter.InputIterator;
import de.uni_koeln.spinfo.tesla.annotation.adapter.queryconstraints.Order;
import de.uni_koeln.spinfo.tesla.annotation.adapter.queryconstraints.Range;
import de.uni_koeln.spinfo.tesla.annotation.adapter.tunguska.DefaultTunguskaOutputAdapter;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.IGeneralizedHeuristicConstituent;
import de.uni_koeln.spinfo.tesla.component.paradigms.data.impl.GeneralizedConstituent;
import de.uni_koeln.spinfo.tesla.roles.core.access.IAnchoredElementAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.core.data.IAnchoredElement;
import de.uni_koeln.spinfo.tesla.roles.filter.access.IFilterAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.filter.data.IFilter;
import de.uni_koeln.spinfo.tesla.roles.parser.access.IConstituentAccessAdapter;
import de.uni_koeln.spinfo.tesla.roles.parser.impl.tunguska.access.TunguskaConstituentAccessAdapter.DefaultTunguskaConstituentAccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.Result;
import de.uni_koeln.spinfo.tesla.runtime.TeslaComponent;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.AccessAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Author;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Component;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Configuration;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Description;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.OutputAdapter;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.RoleDescription;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.Run;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.ServerSubclassChoice;
import de.uni_koeln.spinfo.tesla.runtime.component.annotations.ThreadMode;
import de.uni_koeln.spinfo.tesla.runtime.execution.ICallback;
import de.uni_koeln.spinfo.tesla.runtime.execution.InOrderExecutor;
import de.uni_koeln.spinfo.tesla.runtime.persistence.Annotation;

@Component(threadMode=ThreadMode.NOT_SUPPORTED, 
		author=@Author(	author="Stephan Schwiebert", 
						email="sschwieb@spinfo.uni-koeln.de", 
						web="http://www.spinfo.phil-fak.uni-koeln.de/sschwieb.html", 
						organization="Sprachliche Informationsverarbeitung"),
		description=@Description(name="Suffix Select",
								 summary="Test select method",
								 bigO="unknown",
								 version="1.0",
								 reusableResults=true))
public class SuffixSelect extends TeslaComponent {

	private static final long serialVersionUID = 879812013823937578L;

	
	@AccessAdapter(role="de.uni_koeln.spinfo.tesla.roles.structure.bootstrapping.GeneralizedHeuristicConstituentDetector", name="Input")
	private IConstituentAccessAdapter<IGeneralizedHeuristicConstituent> hypotheses;
	
	@OutputAdapter(dataObject=GeneralizedConstituent.class, type=DefaultTunguskaOutputAdapter.ProtoStuff.class, name="Constituents", 
			accessAdapterImpl=DefaultTunguskaConstituentAccessAdapter.class,
			description="All constituents detected by the parser")
	@RoleDescription("de.uni_koeln.spinfo.tesla.roles.structure.bootstrapping.GeneralizedHeuristicConstituentDetector")
	private IOutputAdapter<GeneralizedConstituent> constituentOut;

	 @Configuration(editor="de.uni_koeln.spinfo.tesla.client.ui.editors.form.configurations.itemeditors.ServerChoiceEditor",
			    defaultValue="de.uni_koeln.spinfo.tesla.component.paradigms.select.RandomizedSelectAlgorithm",
	    		name="Select algorithm", restriction=ServerSubclassChoice.class,
	    		description="The hypothesis selection strategy.", 
	    		evaluateOnServer=true, 
	    		editorConfiguration={"de.uni_koeln.spinfo.tesla.component.paradigms.select.ISelectAlgorithm", "single", "class"})
	private String selectAlgorithm = RandomizedSelectAlgorithm.class.getName();
	
	 @Configuration(editor="de.uni_koeln.spinfo.tesla.client.ui.editors.form.configurations.itemeditors.ServerChoiceEditor",
			    defaultValue="de.uni_koeln.spinfo.tesla.component.paradigms.select.SimpleWeightCalculator",
	    		name="Hypothesis Weight Calculator", restriction=ServerSubclassChoice.class,
	    		description="The algorithm to compute the weight of a hypothesis.", 
	    		evaluateOnServer=true, 
	    		editorConfiguration={"de.uni_koeln.spinfo.tesla.component.paradigms.select.IWeightCalculator", "single", "class"})
	private String weightCalculatorClass = SimpleWeightCalculator.class.getName();

	@AccessAdapter(role="de.uni_koeln.spinfo.tesla.roles.filter.Filter", name="Filters", min=0, max=-1)
	private List<IFilterAccessAdapter<IAnchoredElement, IFilter<IAnchoredElement>>> filters;
	
	@AccessAdapter(role = "de.uni_koeln.spinfo.tesla.roles.core.AnchoredElementGenerator", 
			name = "Sequences", description = "The sequences to align (for instance, sentences)")
	private IAnchoredElementAccessAdapter<IAnchoredElement> sequences;
	
	@Run
	public Result run() throws Exception {
		InputIterator<String> signals = sequences.getAllSignalIds();
		signals.registerAsProgressBar(this);
		final ISelectAlgorithm selector = (ISelectAlgorithm) Thread.currentThread().getContextClassLoader().loadClass(selectAlgorithm).newInstance();
		IWeightCalculator weightCalculator = (IWeightCalculator) Thread.currentThread().getContextClassLoader().loadClass(weightCalculatorClass).newInstance();
		selector.setWeightCalculator(weightCalculator);
		selector.setAnnotationFactory(constituentOut.getAnnotationFactory());
		int counter = 0;
		ICallback<List<Annotation<IGeneralizedHeuristicConstituent>>> callback = new ICallback<List<Annotation<IGeneralizedHeuristicConstituent>>>() {

			@Override
			public void processed(
					List<Annotation<IGeneralizedHeuristicConstituent>> t) {
				for (Annotation<IGeneralizedHeuristicConstituent> annotation : t) {
					constituentOut.store((Annotation<? extends GeneralizedConstituent>) annotation);					
				}
			}

			@Override
			public void exception(ExecutionException e) {
				logger.error("Failure in async processing", e);
			}
		};
		InOrderExecutor<List<Annotation<IGeneralizedHeuristicConstituent>>> executor = new InOrderExecutor<List<Annotation<IGeneralizedHeuristicConstituent>>>(callback);
		while(signals.hasNext()) {
			String signalId = signals.next();
			InputIterator<Annotation<IAnchoredElement>> allSequences = sequences.getAnchoredElements(Range.forSignal(signalId), Order.ORDER_LEFTANCHOR_ASC, IAnchoredElement.class);
			while(allSequences.hasNext()) {
				Annotation<IAnchoredElement> sequence = allSequences.next();
				final List<Annotation<IGeneralizedHeuristicConstituent>> input = hypotheses.getAnnotations(sequence.getRange(false), null).toList();
				Callable<List<Annotation<IGeneralizedHeuristicConstituent>>> callable = new Callable<List<Annotation<IGeneralizedHeuristicConstituent>>>() {

					@Override
					public List<Annotation<IGeneralizedHeuristicConstituent>> call() throws Exception {
						List<Annotation<IGeneralizedHeuristicConstituent>> hypos = filterHypotheses(input);
						hypos = selector.resolveOverlaps(hypos);
						DefaultTunguskaOutputAdapter.sort(hypos);
						return hypos;
					}
					
				};
				executor.enqueue(callable);
				counter++;
				if(counter % 1000 == 0) {
					logger.info("Processed " + counter + " sequences.");
				}
			}
		}
		executor.shutdown();
		
		
		return cancelled() ? Result.CANCELLED : Result.OK;
	}

	

	private List<Annotation<IGeneralizedHeuristicConstituent>> filterHypotheses(
			List<Annotation<IGeneralizedHeuristicConstituent>> hypos) {
		List<Annotation<IGeneralizedHeuristicConstituent>> toReturn = new ArrayList<Annotation<IGeneralizedHeuristicConstituent>>();
		for (Annotation<IGeneralizedHeuristicConstituent> anno : hypos) {
			boolean accepted = true;
			for(IFilterAccessAdapter filter : filters) {
				if(!filter.matches(anno)) {
					accepted = false;
					break;
				}
			}
			if(accepted) {
				toReturn.add(anno);
			}
		}
		return toReturn;
	}

}
